(function (wp) {
  "use strict";

  var el = wp.element.createElement;
  var Fragment = wp.element.Fragment;
  var useState = wp.element.useState;
  var addFilter = wp.hooks.addFilter;
  var createHigherOrderComponent = wp.compose.createHigherOrderComponent;
  var BlockControls = wp.blockEditor.BlockControls;
  var ToolbarGroup = wp.components.ToolbarGroup;
  var ToolbarButton = wp.components.ToolbarButton;
  var Modal = wp.components.Modal;
  var Button = wp.components.Button;
  var TextareaControl = wp.components.TextareaControl;
  var SelectControl = wp.components.SelectControl;
  var Spinner = wp.components.Spinner;
  var apiFetch = wp.apiFetch;

  console.log("Srizon AI Image: Script starting...");

  var MODEL_OPTIONS = [
    {
      label: "FLUX.2 Dev (Cheapest - Around $0.01/image)",
      value: "runware:400@1",
      pricing: "https://runware.ai/pricing",
    },
    {
      label: "Nano Banana (Google - around $0.04/image)",
      value: "google:4@1",
      pricing: "https://runware.ai/pricing",
    },
    {
      label: "GPT Image 1.5 (around $0.04/image for Medium quality)",
      value: "openai:4@1",
      pricing: "https://runware.ai/pricing",
    },
    {
      label: "Midjourney V7 (Midjourney - around $0.10/four images)",
      value: "midjourney:3@1",
      pricing: "https://runware.ai/pricing",
    },
    {
      label: "Nano Banana 2 Pro (Google - expensive $0.138/image)",
      value: "google:4@2",
      pricing: "https://runware.ai/pricing",
    },
  ];

  var STORAGE_KEY = "srizon_ai_image_last_model";

  function getLastModel() {
    try {
      var saved = localStorage.getItem(STORAGE_KEY);
      if (
        saved &&
        MODEL_OPTIONS.some(function (m) {
          return m.value === saved;
        })
      ) {
        return saved;
      }
    } catch (e) {}
    return MODEL_OPTIONS[0].value;
  }

  function saveLastModel(model) {
    try {
      localStorage.setItem(STORAGE_KEY, model);
    } catch (e) {}
  }

  // Model parameter configurations by provider
  // { supportsSteps, supportsCfg, supportsQuality, stepsDefault, stepsMin, stepsMax, cfgDefault, cfgMin, cfgMax }
  var MODEL_PARAMS = {
    runware: {
      supportsSteps: true,
      supportsCfg: true,
      supportsQuality: false,
      stepsDefault: 20,
      stepsMin: 1,
      stepsMax: 50,
      cfgDefault: 4,
      cfgMin: 1,
      cfgMax: 20,
    },
    civitai: {
      supportsSteps: true,
      supportsCfg: true,
      supportsQuality: false,
      stepsDefault: 30,
      stepsMin: 1,
      stepsMax: 50,
      cfgDefault: 7,
      cfgMin: 1,
      cfgMax: 20,
    },
    google: {
      supportsSteps: false,
      supportsCfg: false,
      supportsQuality: false,
    },
    openai: { supportsSteps: false, supportsCfg: false, supportsQuality: true },
    midjourney: {
      supportsSteps: false,
      supportsCfg: false,
      supportsQuality: false,
    },
  };

  var QUALITY_OPTIONS = [
    { label: "High", value: "high" },
    { label: "Medium", value: "medium" },
    { label: "Low", value: "low" },
    { label: "Auto", value: "auto" },
  ];

  function getProviderFromModel(model) {
    return model.split(":")[0];
  }

  function getModelParams(model) {
    var provider = getProviderFromModel(model);
    return (
      MODEL_PARAMS[provider] || { supportsSteps: false, supportsCfg: false }
    );
  }

  function isMidjourneyModel(model) {
    return getProviderFromModel(model) === "midjourney";
  }

  // Model-specific dimension presets
  // Sorted: Wide first (widest to narrowest), then Square, then Tall (narrowest to tallest)
  var MODEL_DIMENSIONS = {
    // FLUX.2 Dev - Min 64x64, max 4MP, multiples of 16
    "runware:400@1": [
      { label: "Ultra Wide 21:9 (1792x768)", value: "1792x768" },
      { label: "Wide 16:9 (1792x1024)", value: "1792x1024" },
      { label: "Landscape 3:2 (1536x1024)", value: "1536x1024" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Portrait 2:3 (1024x1536)", value: "1024x1536" },
      { label: "Tall 9:16 (1024x1792)", value: "1024x1792" },
      { label: "Ultra Tall 9:21 (768x1792)", value: "768x1792" },
    ],
    // Nano Banana
    "google:4@1": [
      { label: "Ultra Wide (1536x672)", value: "1536x672" },
      { label: "Wide 16:9 (1344x768)", value: "1344x768" },
      { label: "Landscape 3:2 (1248x832)", value: "1248x832" },
      { label: "Landscape 4:3 (1184x864)", value: "1184x864" },
      { label: "Landscape 5:4 (1152x896)", value: "1152x896" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Portrait 4:5 (896x1152)", value: "896x1152" },
      { label: "Portrait 3:4 (864x1184)", value: "864x1184" },
      { label: "Portrait 2:3 (832x1248)", value: "832x1248" },
      { label: "Tall 9:16 (768x1344)", value: "768x1344" },
    ],
    // Nano Banana 2 Pro
    "google:4@2": [
      { label: "Ultra Wide (1584x672)", value: "1584x672" },
      { label: "Wide 16:9 (1376x768)", value: "1376x768" },
      { label: "Landscape 3:2 (1264x848)", value: "1264x848" },
      { label: "Landscape 4:3 (1200x896)", value: "1200x896" },
      { label: "Landscape 5:4 (1152x928)", value: "1152x928" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Square 2K (2048x2048)", value: "2048x2048" },
      { label: "Square 4K (4096x4096)", value: "4096x4096" },
      { label: "Portrait 4:5 (928x1152)", value: "928x1152" },
      { label: "Portrait 3:4 (896x1200)", value: "896x1200" },
      { label: "Portrait 2:3 (848x1264)", value: "848x1264" },
      { label: "Tall 9:16 (768x1376)", value: "768x1376" },
    ],
    // GPT Image 1.5
    "openai:4@1": [
      { label: "Landscape 3:2 (1536x1024)", value: "1536x1024" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Portrait 2:3 (1024x1536)", value: "1024x1536" },
    ],
    // DALL-E 3
    "openai:2@3": [
      { label: "Wide 16:9 (1792x1024)", value: "1792x1024" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Tall 9:16 (1024x1792)", value: "1024x1792" },
    ],
    // Midjourney V7
    "midjourney:3@1": [
      { label: "Cinematic 21:9 (1680x720)", value: "1680x720" },
      { label: "Wide 16:9 (1456x816)", value: "1456x816" },
      { label: "Landscape 3:2 (1344x896)", value: "1344x896" },
      { label: "Landscape 4:3 (1232x928)", value: "1232x928" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Portrait 3:4 (928x1232)", value: "928x1232" },
      { label: "Portrait 2:3 (896x1344)", value: "896x1344" },
      { label: "Tall 9:16 (816x1456)", value: "816x1456" },
      { label: "Cinematic Vertical 9:21 (720x1680)", value: "720x1680" },
    ],
  };

  function getDimensionsForModel(model) {
    return MODEL_DIMENSIONS[model] || MODEL_DIMENSIONS["runware:400@1"];
  }

  function getDefaultDimensionForModel(model) {
    var dims = getDimensionsForModel(model);
    // Find Square 1:1 as default, or use first option
    for (var i = 0; i < dims.length; i++) {
      if (dims[i].value === "1024x1024") {
        return dims[i].value;
      }
    }
    return dims[0].value;
  }

  function AIImageModal(props) {
    var isOpen = props.isOpen;
    var onClose = props.onClose;
    var onImageGenerated = props.onImageGenerated;

    var _prompt = useState("");
    var prompt = _prompt[0];
    var setPrompt = _prompt[1];

    var _model = useState(getLastModel());
    var selectedModel = _model[0];
    var setSelectedModel = function (model) {
      saveLastModel(model);
      _model[1](model);
      // Reset dimension to default for the new model
      var newDefault = getDefaultDimensionForModel(model);
      setDimensionPreset(newDefault);
    };

    var _dim = useState(getDefaultDimensionForModel(getLastModel()));
    var dimensionPreset = _dim[0];
    var setDimensionPreset = _dim[1];

    var _steps = useState(0);
    var steps = _steps[0];
    var setSteps = _steps[1];

    var _cfg = useState(0);
    var cfgScale = _cfg[0];
    var setCfgScale = _cfg[1];

    var _quality = useState("high");
    var quality = _quality[0];
    var setQuality = _quality[1];

    var _gen = useState(false);
    var isGenerating = _gen[0];
    var setIsGenerating = _gen[1];

    var _err = useState("");
    var error = _err[0];
    var setError = _err[1];

    var modelParams = getModelParams(selectedModel);

    if (!isOpen) return null;

    function handleGenerate() {
      if (!prompt.trim()) {
        setError("Please enter a prompt.");
        return;
      }
      setError("");
      setIsGenerating(true);

      var parts = dimensionPreset.split("x");
      var width = parseInt(parts[0], 10);
      var height = parseInt(parts[1], 10);

      var requestData = {
        prompt: prompt.trim(),
        width: width,
        height: height,
        model: selectedModel,
      };

      if (modelParams.supportsSteps && steps > 0) {
        requestData.steps = steps;
      }
      if (modelParams.supportsCfg && cfgScale > 0) {
        requestData.cfg_scale = cfgScale;
      }
      if (modelParams.supportsQuality) {
        requestData.quality = quality;
      }

      apiFetch({
        path: "/srizon-ai-image/v1/generate-image",
        method: "POST",
        data: requestData,
      })
        .then(function (response) {
          setIsGenerating(false);
          if (response.success && response.data) {
            onImageGenerated(response.data);
            onClose();
            setPrompt("");
          } else {
            setError(response.message || "Failed to generate image.");
          }
        })
        .catch(function (err) {
          setIsGenerating(false);
          setError(err.message || "An error occurred.");
        });
    }

    var currentModelOption = MODEL_OPTIONS.find(function (m) {
      return m.value === selectedModel;
    });
    var pricingUrl = currentModelOption ? currentModelOption.pricing : "#";

    return el(
      Modal,
      {
        title: "Generate with Srizon AI Image",
        onRequestClose: onClose,
        className: "srizon-ai-image-modal",
      },
      el(
        "div",
        { className: "srizon-ai-image-modal-content" },
        el(TextareaControl, {
          label: "Image Prompt",
          value: prompt,
          onChange: setPrompt,
          placeholder: "Describe the image you want to generate...",
          rows: 4,
          disabled: isGenerating,
        }),
        el(
          "div",
          { className: "srizon-ai-image-model-row" },
          el(SelectControl, {
            label: "Model",
            value: selectedModel,
            options: MODEL_OPTIONS.map(function (m) {
              return { label: m.label, value: m.value };
            }),
            onChange: setSelectedModel,
            disabled: isGenerating,
          }),
          el(
            "a",
            {
              href: pricingUrl,
              target: "_blank",
              rel: "noopener noreferrer",
              className: "srizon-ai-image-pricing-link",
            },
            "View Pricing ↗"
          )
        ),
        isMidjourneyModel(selectedModel) &&
          el(
            "div",
            { className: "srizon-ai-image-info-note" },
            "Note: Midjourney generates 4 images per request. All images will be saved to your Media Library."
          ),
        el(SelectControl, {
          label: "Image Dimensions",
          value: dimensionPreset,
          options: getDimensionsForModel(selectedModel),
          onChange: setDimensionPreset,
          disabled: isGenerating,
          help: "Higher dimensions will increase generation price.",
        }),
        (modelParams.supportsSteps || modelParams.supportsCfg) &&
          el(
            "div",
            { className: "srizon-ai-image-advanced-params" },
            el(
              "div",
              { className: "srizon-ai-image-params-label" },
              "Advanced Parameters"
            ),
            el(
              "div",
              { className: "srizon-ai-image-params-row" },
              modelParams.supportsSteps &&
                el(
                  "div",
                  { className: "srizon-ai-image-param-input" },
                  el("label", null, "Steps"),
                  el("input", {
                    type: "number",
                    value: steps || modelParams.stepsDefault,
                    min: modelParams.stepsMin,
                    max: modelParams.stepsMax,
                    step: 1,
                    disabled: isGenerating,
                    onChange: function (e) {
                      setSteps(parseInt(e.target.value, 10));
                    },
                  }),
                  el(
                    "span",
                    { className: "srizon-ai-image-param-hint" },
                    modelParams.stepsMin + "-" + modelParams.stepsMax
                  ),
                  el(
                    "span",
                    { className: "srizon-ai-image-param-help" },
                    "Controls quality and refinement (how \"finished\" the image looks). Higher values increase price."
                  )
                ),
              modelParams.supportsCfg &&
                el(
                  "div",
                  { className: "srizon-ai-image-param-input" },
                  el("label", null, "CFG Scale"),
                  el("input", {
                    type: "number",
                    value: cfgScale || modelParams.cfgDefault,
                    min: modelParams.cfgMin,
                    max: modelParams.cfgMax,
                    step: 0.5,
                    disabled: isGenerating,
                    onChange: function (e) {
                      setCfgScale(parseFloat(e.target.value));
                    },
                  }),
                  el(
                    "span",
                    { className: "srizon-ai-image-param-hint" },
                    modelParams.cfgMin + "-" + modelParams.cfgMax
                  ),
                  el(
                    "span",
                    { className: "srizon-ai-image-param-help" },
                    "Controls creativity vs. obedience (how strictly it follows your prompt)"
                  )
                )
            )
          ),
        modelParams.supportsQuality &&
          el(
            "div",
            { className: "srizon-ai-image-advanced-params" },
            el(
              "div",
              { className: "srizon-ai-image-params-label" },
              "Quality Settings"
            ),
            el(SelectControl, {
              label: "Quality",
              value: quality,
              options: QUALITY_OPTIONS,
              onChange: setQuality,
              disabled: isGenerating,
              help: "Higher quality takes longer and costs more.",
            })
          ),
        error && el("div", { className: "srizon-ai-image-error" }, error),
        el(
          "div",
          { className: "srizon-ai-image-modal-actions" },
          el(
            Button,
            { variant: "secondary", onClick: onClose, disabled: isGenerating },
            "Cancel"
          ),
          el(
            Button,
            {
              variant: "primary",
              onClick: handleGenerate,
              disabled: isGenerating || !prompt.trim(),
            },
            isGenerating ? "Generating..." : "Generate Image"
          )
        )
      )
    );
  }

  var withAIImageButton = createHigherOrderComponent(function (BlockEdit) {
    return function (props) {
      if (props.name !== "core/image") {
        return el(BlockEdit, props);
      }

      var _modal = useState(false);
      var isModalOpen = _modal[0];
      var setIsModalOpen = _modal[1];

      function handleImageGenerated(imageData) {
        props.setAttributes({
          id: imageData.attachment_id,
          url: imageData.url,
          alt: imageData.alt || "",
        });
      }

      return el(
        Fragment,
        null,
        el(BlockEdit, props),
        el(
          BlockControls,
          null,
          el(
            ToolbarGroup,
            null,
            el(
              Button,
              {
                icon: "images-alt2",
                variant: "tertiary",
                onClick: function () {
                  setIsModalOpen(true);
                },
                className: "srizon-ai-image-toolbar-btn",
              },
              "Generate with AI"
            )
          )
        ),
        el(AIImageModal, {
          isOpen: isModalOpen,
          onClose: function () {
            setIsModalOpen(false);
          },
          onImageGenerated: handleImageGenerated,
        })
      );
    };
  }, "withAIImageButton");

  addFilter(
    "editor.BlockEdit",
    "srizon-ai-image/with-ai-image-button",
    withAIImageButton
  );

  console.log("Srizon AI Image: Filter registered successfully");
})(window.wp);
